/**
  *******************************************************************************
  * 
  * @file    hal_base.c
  * @brief   ES7P203 base module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The base driver can be used as follows:
    (+) Initialize ram by using ram_clear().
    (+) Delay time by using delay()
    (+) Initialize clock by using clock_init()
    (+) Get system clock by using get_system_clock()
    (+) Get reset flag by using get_reset_flag()
    (+) Clear reset flag by using clear_reset_flag()
    (+) Multiple calculate by using mul_calc()
    (+) Divide calculate by using div_calc()

  @endverbatim
  *********************************************************************************
  */
#include "hal_base.h"


/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup BASE base
  * @brief base module driver
  * @{
  */

/**
  * @defgroup Base_Public_Functions base Public Functions
  * @brief    Base Functions
@verbatim
=================================================================================
		##### Base Public Functions #####
=================================================================================
  [..]  The base driver can be used as follows:
    (+) Initialize ram by using ram_clear().
    (+) Delay time by using delay()
    (+) Initialize clock by using clock_init()
    (+) Get system clock by using get_system_clock()
    (+) Get reset flag by using get_reset_flag()
    (+) Clear reset flag by using clear_reset_flag()
    (+) Multiple calculate by using mul_calc()
    (+) Divide calculate by using div_calc()
@endverbatim
  * @{
  */

/**
  * @brief  Initialize ram
  * @param  None.
  * @retval None.
  */
void ram_clear(void)
{
    for (IAAH=0; IAAH<=0x03; IAAH++)
    {
        for (IAAL=0; IAAL<0xFF; IAAL++)
            IAD = 0x00;
        IAD = 0x00;
    }
}

/**
  * @brief  Delay time
  * @param  times:delay times.
  * @retval None.
  */
void delay(uint16_t times)
{
	uint16_t loop = 0;
	for (; loop < times; loop++) {
		__asm{
			NOP
			NOP
			NOP
			NOP
			NOP
			NOP
			NOP
			NOP
			NOP
			NOP
		}
	}

}

/**
  * @brief  Initialize clock
  * @param  src: the source of clock
  * @param  div: divide clock 
  * @param  clk_out: clock output to pin function
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t clock_init(clock_src_t src, clock_div_t div, type_func_t clk_out)
{
	assert_param(IS_CLOCK_SRC_TYPE(src));
	assert_param(IS_CLOCK_DIV_TYPE(div));
	
	CLOCK_UNLOCK();
	
	if (src == CLOCK_SRC_HRC)
		HRCEN = 1;
	
	SCKS  = src;
	DIVS0 = div;
	DIVS1 = (div >> 1);
	DIVS2 = (div >> 2);
	
	while (CHG == 1);
	
	if (SST != src)
		return ERROR;
	
	CLKOEN = clk_out;
	
	CLOCK_LOCK();
	return OK;
		
}

/**
  * @brief  Get system clock
  * @param  None.
  * @retval clock value.
  */
uint32_t get_system_clock(void)
{
	if (SCKS == 0)
		return 16000000;
	else
		return 32000;
}

/**
  * @brief  Initialize ram
  * @param  flag: Reset of flag
  * @retval The flag status.
  */
flag_status_t get_reset_flag(reset_flag_t flag)
{
	return (PWRC & flag);
}

/**
  * @brief  Clear reset flag
  * @param  flag: Reset of flag
  * @retval None.
  */
void clear_reset_flag(reset_flag_t flag)
{
	PWRC |= flag;
	return;
}

/**
  * @brief  Multiple calculate
  * @param  mul_a: Multiplier a
  * @param  mul_b: Multiplier b
  * @retval Calculate data
  */
uint16_t mul_calc(uint8_t mul_a, uint8_t mul_b)
{
	MULA = mul_a;
	MULB = mul_b;
	delay(1);
	
	return ((MULH << 8) | (MULL));
}

/**
  * @brief  Divide calculate
  * @param  divider: The divider
  * @param  divisor: The divisor
  * @param  remainder: The remainder
  * @retval Calculate data
  */
uint32_t div_calc(uint32_t divider, uint16_t divisor, uint16_t *remainder)
{
	DIVE0 = divider;
	DIVE1 = divider >> 8;
	DIVE2 = divider >> 16;
	DIVE3 = divider >> 24;
	DIVSL = divisor;
	DIVSH = divisor >> 8;

	VGO = 1;

	while (VGO == 1) {
		if (VERR = 1)
			return 0;
	}

	*remainder = ((DIVRH << 8) | DIVRL);

	return ((DIVQ3 << 24) | (DIVQ2 << 16) | (DIVQ1 << 8) | DIVQ0);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
